<?php
$theme_name = "Harmonix";
global $theme_name;

/**
 * Harmonix functions and definitions
 *
 * @package Harmonix
 * @since Harmonix 1.0.0
 */

 if ( ! function_exists( 'harmonix_support' ) ) :
	function harmonix_support() {
		// Load Theme Textdomain.
		load_theme_textdomain( 'harmonix', get_template_directory() . '/languages' );
		if ( ! 'harmonix' === wp_get_theme()->get( 'harmonix' ) ) {
			load_theme_textdomain( wp_get_theme()->get( 'harmonix' ) );
		}
	}

	// Enqueue editor styles.
	add_editor_style(
		array(
			'/assets/ponyfill.css',
		)
	);

	add_action( 'after_setup_theme', 'harmonix_support' );
endif;

// Theme custom block styles.
function harmonix_custom_gutenberg_blocks() {
	register_block_style(
		'core/list',
		array(
			'name'         => 'bullet-checked',
			'label'        => 'Bullet Checked',
		)
	);

	register_block_style(
			'core/list',
			array(
				'name'         => 'bullet-x',
				'label'        => 'Bullet X',
			)
	);

	register_block_style(
		'core/button',
			array(
				'name'  => 'button-arrow',
				'label' => __( 'Arrow', 'harmonix' ),
			)
	);
}
add_action( 'init', 'harmonix_custom_gutenberg_blocks' );

/**
 * Custom Template part.
 *
 * @param array $areas Template part areas.
 *
 * @return array
 */
function harmonix_custom_template_part_area( $areas ) {
	array_push(
		$areas,
		array(
			'area'        => 'query',
			'label'       => esc_html__( 'Query', 'harmonix' ),
			'description' => esc_html__( 'Custom query area', 'harmonix' ),
			'icon'        => 'layout',
			'area_tag'    => 'div',
		)
	);

	return $areas;
}
add_filter( 'default_wp_template_part_areas', 'harmonix_custom_template_part_area' );

if ( ! function_exists( 'harmonix_load_scripts' ) ) {
	/**
	 * Enqueue CSS Stylesheets and Javascript files.
	 *
	 * @return void
	 */
	function harmonix_load_scripts() {
		$harmonix_theme_version = wp_get_theme()->get( 'Version' );

		// Styles
		wp_enqueue_style( 'harmonix-ponyfill', get_template_directory_uri() . '/assets/ponyfill.css', array(), $harmonix_theme_version );

		if ( is_rtl() ) {
			wp_enqueue_style( 'rtl', get_template_directory_uri() . '/assets/rtl.css', array(), $harmonix_theme_version, 'all' );
		}

		// Scripts
		wp_enqueue_script( 'harmonix-call-scripts', get_template_directory_uri() . '/assets/js/common.js', array( 'jquery', 'wp-i18n' ), $harmonix_theme_version, true );

		wp_enqueue_style('dashicons');
	}
	add_action( 'wp_enqueue_scripts', 'harmonix_load_scripts' );
}

// Add admin CSS that loads only on the themes page
function pup_load_admin_css_on_themes_page($hook_suffix) {
    if ($hook_suffix === 'themes.php') {
        wp_enqueue_style( 'puplaza-admin-css', get_template_directory_uri() . '/assets/admin.css', array(), '1.0', 'all' );
    }
}
add_action('admin_enqueue_scripts', 'pup_load_admin_css_on_themes_page');

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
if ( ! function_exists( 'harmonix_body_classes' ) ) :
	function harmonix_body_classes( $classes ) {

		// Adds a class of tc-theme-frontend when viewing frontend.
		if ( !is_admin() ) {
			$classes[] = esc_attr( 'tc-theme-frontend' );
		}

		return $classes;
	}
	add_filter( 'body_class', 'harmonix_body_classes' );
endif;

// Block Patterns.
require get_template_directory() . '/inc/block-patterns.php';

// Add the child theme patterns if they exist.
if ( file_exists( get_stylesheet_directory() . '/inc/block-patterns.php' ) ) {
	require_once get_stylesheet_directory() . '/inc/block-patterns.php';
}

// Load plugin activation script
require get_template_directory() . '/inc/plugin-activation.php';

// Load Merlin
require_once get_parent_theme_file_path( '/inc/merlin/vendor/autoload.php' );
require_once get_parent_theme_file_path( '/inc/merlin/class-merlin.php' );
require_once get_parent_theme_file_path( '/inc/merlin-config.php' );

/**
 * Filter callback function to lengthen the excerpt.
 *
 * This function is used as a callback for the 'tc_ef_long_excerpt' filter hook.
 * It returns the desired length of the excerpt, which is 80 characters in this case.
 *
 * @return int The desired length of the excerpt.
 */
function harmonix_lenghten_excerpt() {
	return 80;
}
add_filter('tc_ef_long_excerpt', 'harmonix_lenghten_excerpt');


/**
 * Filter callback function to modify the length of the short excerpt.
 *
 * @return int The desired length of the short excerpt.
 */
function harmonix_lenghten_excerpt_short() {
	return 30;
}
add_filter('tc_ef_short_excerpt', 'harmonix_lenghten_excerpt_short');

/**
 * Filter callback function for 'tc_ef_2_col_excerpt' filter.
 * Returns the desired length of the excerpt in characters.
 *
 * @return int The desired length of the excerpt.
 */
function harmonix_tc_ef_2_col_excerpt() {
	return 60;
}
add_filter('tc_ef_2_col_excerpt', 'harmonix_tc_ef_2_col_excerpt');


/**
 * Filter callback function for 'tc_ef_3_col_excerpt' filter.
 * Returns the desired length for the 3 column excerpt.
 *
 * @return int The desired length for the 3 column excerpt.
 */
function harmonix_tc_ef_3_col_excerpt() {
	return 45;
}
add_filter('tc_ef_3_col_excerpt', 'harmonix_tc_ef_3_col_excerpt');

/**
 * Removes the 'child' step from Merlin setup process
 *
 * @param array $steps The array of steps.
 * @return array The modified array of steps.
 */
function harmonix_tc_remove_child_step( $steps ) {
	unset( $steps['child'] );
	return $steps;
}
add_filter( get_template() . '_merlin_steps', 'harmonix_tc_remove_child_step');

/**
 * Set up import files for Merlin
 *
 * This function returns an array of import file details for the Harmonix theme demo import.
 *
 * @return array An array of import file details.
 */
if ( ! function_exists( 'harmonix_merlin_local_import_files' ) ) :
	function harmonix_merlin_local_import_files() {
		return array(
			array(
				'import_file_name'             => 'Harmonix Demo',
				'import_file_url'              => 'https://themetick.com/importer-files/harmonix-wp.xml',
				'import_preview_image_url'     => 'https://themetick.com/importer-files/harmonix-screenshot.png',
				'import_notice'                => __( 'Thank you!', 'harmonix' ),
				'preview_url'                  => 'https://themetick.com',
			),
		);
	}
	add_filter( 'merlin_import_files', 'harmonix_merlin_local_import_files' );
endif;

/**
 * Filters the WooCommerce enable setup wizard option.
 *
 * Disable woocommerce plugin activation redirect
 *
 * @param bool $true The original value of the WooCommerce enable setup wizard option.
 * @return bool The modified value of the WooCommerce enable setup wizard option.
 */
function harmonix_tc_filter_woocommerce_enable_setup_wizard( $true ) {
	return false;
}
add_filter( 'woocommerce_enable_setup_wizard', 'harmonix_tc_filter_woocommerce_enable_setup_wizard', 10, 1 );

/**
 * Adds a custom class to the WooCommerce product gallery based on the presence of images.
 *
 * @param array $classes An array of CSS classes for the product.
 * @param WC_Product $product The WooCommerce product object.
 * @return array The modified array of CSS classes.
 */
function harmonix_woocommerce_gallery_check( $classes, $product ) {
	$attachment_ids = $product->get_gallery_image_ids();

	if ( empty($attachment_ids) ) {
		$additional_class = 'no-image-gallery';
	} else {
		$additional_class = '';
	}

	$classes[] = $additional_class;

	return $classes;
}
add_filter('woocommerce_post_class', 'harmonix_woocommerce_gallery_check', 10, 2);

// Turn on the Customizer
add_action( 'customize_register', '__return_true' );

/**
 * This function automatically sets the 'show_tickets_automatically' meta key to '1'
 * for posts that use the 'tickera-event-singular' page template.
 */
function harmonix_set_show_tickets_automatically_meta( $null, $object_id, $meta_key, $single ) {
    if ( $meta_key === 'show_tickets_automatically' ) {
        $template = get_post_meta( $object_id, '_wp_page_template', true );
        if ( $template === 'tickera-event-singular' ) {
            return '1';
        }
    }
    return $null;
}
add_filter('get_post_metadata', 'harmonix_set_show_tickets_automatically_meta', 10, 4);

/**
 * Filter the blog page title from your demo content.
 * If your demo's blog page title is "Blog", you don't need this.
 */
if ( ! function_exists( 'harmonix_merlin_content_blog_page_title' ) ) :
	function harmonix_merlin_content_blog_page_title( $output ) {
		return 'News';
	}
add_filter( 'merlin_content_blog_page_title', 'harmonix_merlin_content_blog_page_title' );
endif;

/**
 * Admin notice to finish theme installation wizard
 */
if ( ! function_exists( 'harmonix_return_to_merlin' ) ) :
	function harmonix_return_to_merlin() {
		if ( ! get_option( 'merlin_harmonixwp_completed', false ) ) {
			global $theme_name;

			echo wp_kses_post( sprintf(
			/* translators: %s: Tickera */
				__( '<div class="notice notice-warning"><p><strong>%s</strong> is nearly ready! Just complete the <a href="%s">installation wizard</a> to finish setting up.</p></div>', 'harmonix' ),
				esc_html( $theme_name ),
				esc_url( admin_url( '?page=merlin' ) )
			) );
		}

	}
	add_action( 'admin_notices', 'harmonix_return_to_merlin', 999 );
endif;
