<?php

namespace Automattic\WCShipping\Shipments;

use Automattic\WCShipping\Connect\WC_Connect_Service_Settings_Store;
use Automattic\WCShipping\Shipment\ShipmentFromLabelGenerator;
use Automattic\WCShipping\Shipments\Models\Shipments;
use Exception;
use WC_Order;
use Automattic\WCShipping\Exceptions\RESTRequestException;

class ShipmentsService {

	const META_KEY = '_wcshipping-shipments';

	/**
	 * @var WC_Connect_Service_Settings_Store
	 */
	protected $settings_store;

	public function __construct( WC_Connect_Service_Settings_Store $settings_store ) {
		$this->settings_store = $settings_store;
	}

	/**
	 * Update shipments on an order.
	 *
	 * @param int   $order_id The WC order ID.
	 * @param array $raw_shipments The shipments sent with the api call.
	 * @param array $shipment_ids_to_update The shipment ID map to update.
	 * @return RESTRequestException|int
	 *
	 * @throws RESTRequestException Will throw an error if the order is not found.
	 */
	public function update_order_shipments( int $order_id, array $raw_shipments, array $shipment_ids_to_update = array() ) {
		/**
		 * @var WC_Order $order
		 */
		$order = wc_get_order( $order_id );
		if ( ! $order instanceof WC_Order ) {
			/*
			 * We have to escape an exceptions output in case it's not caught internally.
			 *
			 * @link https://github.com/WordPress/WordPress-Coding-Standards/issues/884
			 */
			throw new RESTRequestException( esc_html__( 'Order not found when updating order shipments ', 'woocommerce-shipping' ) );
		}

		try {
			// Cast shipments to the data model to ensure data sanity
			$shipments_model = new Shipments( $raw_shipments );
			$shipments       = $shipments_model->to_array();
		} catch ( Exception $e ) {
			throw new RESTRequestException(
				sprintf(
					/* translators: %s: Reason for the failure in creating the data model */
					esc_html__( 'Shipment model creation failed with error: %s', 'woocommerce-shipping' ),
					esc_html( $e->getMessage() )
				)
			);
		}

		$order->update_meta_data( self::META_KEY, $shipments );

		if ( ! empty( $shipment_ids_to_update ) ) {
			$order_labels = $this->settings_store->get_label_order_meta_data( $order_id );
			foreach ( $order_labels as &$label ) {
				if ( isset( $shipment_ids_to_update[ $label['id'] ] ) ) {
					$label['id'] = $shipment_ids_to_update[ $label['id'] ];
				}
			}
			$order->update_meta_data( 'wcshipping_labels', $order_labels );
		}

		return $order->save();
	}

	/**
	 * Get the order shipments JSON.
	 *
	 * If shipments are already stored as order meta, returns a JSON encoded object of those shipments.
	 * Otherwise, it generates shipments (and flag whether fallback was needed) and then returns a JSON encoded object.
	 *
	 * @param int $order_id The WC order ID.
	 * @return array {
	 *     @type array $shipments The shipments.
	 *     @type array $autogenerated_from_labels The autogenerated from labels.
	 * }
	 */
	public function get_order_shipments_data( $order_id ): array {
		/**
		 * @var WC_Order $order
		 */
		$order = wc_get_order( $order_id );
		if ( ! $order instanceof WC_Order ) {
			return array(
				'shipments'                 => array(),
				'autogenerated_from_labels' => array(),
			);
		}

		$raw_shipments = $order->get_meta( self::META_KEY );

		if ( ! empty( $raw_shipments ) ) {
			try {
				// Cast shipments to the data model to ensure data sanity
				$shipments_model = new Shipments( $raw_shipments );
				$shipments       = $shipments_model->to_array();
			} catch ( Exception $e ) {
				throw new RESTRequestException(
					sprintf(
						/* translators: %s: Reason for the failure in creating the data model */
						esc_html__( 'Shipment model creation failed with error: %s', 'woocommerce-shipping' ),
						esc_html( $e->getMessage() )
					)
				);
			}

			return array(
				'shipments'                 => $shipments,
				'autogenerated_from_labels' => array(),
			);
		}

		$labels = $this->settings_store->get_label_order_meta_data( $order->get_id() );

		// Filter labels that are refunded or ones that errored while purchase
		// when auto-generating shipments to prevent
		// creation of extra shipments from refunded labels.
		$valid_labels = array_filter(
			$labels,
			function ( $label ) {
				return empty( $label['refund'] ) && $label['status'] !== 'PURCHASE_ERROR';
			}
		);

		$generator = new ShipmentFromLabelGenerator( $order );

		return $generator->generate_shipments( $valid_labels );
	}
}
