<?php
namespace ReyCore\ACF;

if (!defined('ABSPATH')) exit; // Exit if accessed directly


class Fields {

	private static $fields = [];

	function __construct(){
		add_action('acf/init', [$this, 'init']);
	}

	function init(){

		if( ! function_exists('acf_add_local_field_group') ){
			return;
		}

		// load predefined fields
		if( ! apply_filters('reycore/acf/use_predefined_fields', ! (defined('REYCORE_ACF_DISABLE_FIELDS') && REYCORE_ACF_DISABLE_FIELDS)) ){
			return;
		}

		self::$fields = [
			'global_sections'    => $this->f__global_sections(),
			'product_settings'   => $this->f__product_settings(),
			'page_settings'      => $this->f__page_settings(),
			'theme_settings'     => $this->f__theme_settings(),
			'icon_menu_settings' => $this->f__icon_menu_settings(),
		];

		$this->prepare_fields();

		do_action('reycore/acf/fields', $this);

		foreach (self::$fields as $key => $fields) {
			acf_add_local_field_group($fields);
		}
	}

	public function prepare_fields(){

		add_filter('acf/prepare_field/key=field_5ecd3db74623a', [$this, 'top_container_padding_placeholders']);
		add_filter('acf/prepare_field/key=field_5ecd3ddd4623c', [$this, 'bottom_container_padding_placeholders']);
		add_filter('acf/prepare_field/key=field_5dbd36274ec3a', [$this, 'container_spacing_placeholder']);

	}

	public function top_container_padding_placeholders($field){
		$field['placeholder'] = \ReyCore\ACF\Helper::get_container_padding_placeholders('top');
		return $field;
	}

	public function bottom_container_padding_placeholders($field){
		$field['placeholder'] = \ReyCore\ACF\Helper::get_container_padding_placeholders('bottom');
		return $field;
	}

	public function container_spacing_placeholder($field){
		$field['placeholder'] = get_theme_mod('container_spacing', 15);
		return $field;
	}

	public function set_group_fields( $group, $new_field, $after = '' ){

		if( ! isset(self::$fields[$group]) ){
			return;
		}

		if( ! $after ){
			array_push(self::$fields[$group]['fields'], $new_field );
		}
		else {
			// find key
			$key = array_search($after, array_column( self::$fields[$group]['fields'], 'name' ) );
			// add to group
			self::$fields[$group]['fields'] = $this->array_insert_after( self::$fields[$group]['fields'], $key, [$new_field] );
		}

	}

	public function array_insert_after( array $array, $key, array $new ) {

		$keys = array_keys( $array );
		$index = array_search( $key, $keys );
		$pos = false === $index ? count( $array ) : $index + 1;

		return array_merge( array_slice( $array, 0, $pos ), $new, array_slice( $array, $pos ) );
	}

	public function f__global_sections(){
		return [
			'key' => 'group_5c4c18df892d1',
			'title' => 'Global Sections',
			'fields' => array(
				array(
					'key' => 'field_5c4c18fb06515',
					'label' => esc_html_x('Choose Type', 'Backend setting label', 'rey-core'),
					'name' => 'gs_type',
					'type' => 'button_group',
					'instructions' => esc_html_x('Headers & Footers need to be manually assigned in Customizer > Header (Footer) > General , or for specific pages, into the page\' custom options. Generic sections can be assigned to different page positions, either in Customizer > General > Generic Sections (for site-wide usage), or specific pages, into the page\' custom options.', 'Backend setting description', 'rey-core'),
					'required' => 1,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'gs-type-selector',
						'id' => '',
					),
					'choices' => array(
						'generic' => 'Generic Section',
						'header' => 'Header',
						'cover' => 'Page Cover',
						'footer' => 'Footer',
					),
					'allow_null' => 0,
					'default_value' => 'generic',
					'layout' => 'horizontal',
					'return_format' => 'value',
				),
			),
			'location' => array(
				array(
					array(
						'param' => 'post_type',
						'operator' => '==',
						'value' => 'rey-global-sections',
					),
				),
			),
			'menu_order' => 0,
			'position' => 'acf_after_title',
			// 'style' => 'default',
			'style' => 'seamless',
			'label_placement' => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen' => '',
			'active' => true,
			'description' => '',
		];
	}

	public function f__product_settings(){

		$product_settings = array(

			array(
				'key' => 'field_5d4ff5734de9f',
				'label' => esc_html_x('Misc.', 'Backend setting label', 'rey-core'),
				'name' => '',
				'type' => 'tab',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'placement' => 'top',
				'endpoint' => 0,
			),

			array(
				'key' => 'field_5d4ff5a94dea0',
				'label' => esc_html_x('Product Info Block/Tab', 'Backend setting label', 'rey-core'),
				'name' => 'product_info',
				'type' => 'select',
				'instructions' => esc_html_x('Select if you want to show or hide the "Information" content, which is displayed after the product summary. The content is pre-defined in "Customizer > WooCommerce > Product Page - Content" panel. You can also select Custom content, and you will be able to override it.', 'Backend setting description', 'rey-core'),
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => 'rey-decrease-list-size',
					'id' => '',
				),
				'choices' => array(
					1 => 'Show',
					2 => 'Hide',
					'custom' => 'Custom',
				),
				'default_value' => false,
				'allow_null' => 1,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',

			),

			array(
				'key' => 'field_5d4ff6c64dea1',
				'label' => esc_html_x('Custom Content', 'Backend setting label', 'rey-core'),
				'name' => 'product_info_content',
				'type' => 'wysiwyg',
				'instructions' => esc_html_x('Override the custom content pre-defined in "Customizer > WooCommerce > Product Page - Content" panel.', 'Backend setting description', 'rey-core'),
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'field_5d4ff5a94dea0',
							'operator' => '==',
							'value' => 'custom',
						),
					),
				),
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'default_value' => '',
				'tabs' => 'all',
				'toolbar' => 'full',
				'media_upload' => 1,
				'delay' => 1,

			),

			array(
				'key' => 'field_60a1063400686',
				'label' => esc_html_x('Specifications Block/Tab', 'Backend setting label', 'rey-core'),
				'name' => 'single_specifications_block',
				'type' => 'true_false',
				'instructions' => esc_html_x('Disable if you don\'t want to display the Specifications block/tab.', 'Backend setting description', 'rey-core'),
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'message' => '',
				'default_value' => 1,
				'ui' => 1,
				'ui_on_text' => 'Show',
				'ui_off_text' => 'Hide',

			),

			array(
				'key' => 'field_5e24805172eb5',
				'label' => esc_html_x('Add to cart', 'Backend setting label', 'rey-core'),
				'name' => '',
				'type' => 'tab',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'placement' => 'top',
				'endpoint' => 0,

			),
			array(
				'key' => 'field_645d804382d23',
				'label' => '',
				'name' => '',
				'type' => 'message',
				'wrapper' => array(
					'class' => '--title-h3',
				),
				'message' => '<h3>Content <strong>Before</strong> Add to cart button</h3><p style="margin-bottom:0;">Select if you want to show some content *Before* the Add to Cart button. You can globally assign content on all products by accessing "Customizer > WooCommerce > Product Page - Components" panel and pick a Global Section or enable the "Text Before Add to cart button" option.</p>',
				'esc_html' => 0,
			),
			array(
				'key' => 'field_6117b9ce17d0c',
				'label' => esc_html_x('Global Section', 'Backend setting label', 'rey-core'),
				'name' => 'gs_before_atc',
				'type' => 'global_sections',
				'instructions' => esc_html_x('Pick a global section', 'Backend setting description', 'rey-core'),
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => 'rey-decrease-list-size',
					'id' => '',
				),
				'rey_export' => 'post_id',
				'default_value' => '',
				'gs_type' => 'generic',
				'choices'     => [
					''  => esc_attr__( '- Inherit -', 'rey-core' ),
					'no'  => esc_attr__( '- Disabled -', 'rey-core' ),
				],
			),
			array(
				'key' => 'field_5e24815a72eb8',
				'label' => esc_html_x('Text', 'Backend setting label', 'rey-core'),
				'name' => 'enable_text_before_add_to_cart',
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => 'rey-decrease-list-size',
					'id' => '',
				),
				'choices' => array(
					'true' => esc_html_x('Show', 'Backend setting label', 'rey-core'),
					'false' => esc_html_x('Hide', 'Backend setting label', 'rey-core'),
					'custom' => esc_html_x('Show Custom Text', 'Backend setting label', 'rey-core'),
				),
				'default_value' => false,
				'allow_null' => 1,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',
			),
			array(
				'key' => 'field_5e24810172eb7',
				'label' => esc_html_x('Custom text before Add to cart button', 'Backend setting label', 'rey-core'),
				'name' => 'text_before_add_to_cart',
				'type' => 'textarea',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'field_5e24815a72eb8',
							'operator' => '==',
							'value' => 'custom',
						),
					),
				),
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'default_value' => '',
				'placeholder' => '',
				'maxlength' => '',
				'rows' => '',
				'new_lines' => 'wpautop',
			),
			array(
				'key' => 'field_643880d245d23',
				'label' => '',
				'name' => '',
				'type' => 'message',
				'wrapper' => array(
					'class' => '--title-h3',
				),
				'message' => '<h3>Content <strong>After</strong> Add to cart button</h3><p style="margin-bottom:0;">Select if you want to show some content *After* the Add to Cart button. You can globally assign content on all products by accessing "Customizer > WooCommerce > Product Page - Components" panel and pick a Global Section or enable the "Text After Add to cart button" option.</p>',
				'esc_html' => 0,
			),
			array(
				'key' => 'field_6b9101717dcec',
				'label' => esc_html_x('Global Section', 'Backend setting label', 'rey-core'),
				'name' => 'gs_after_atc',
				'type' => 'global_sections',
				'instructions' => esc_html_x('Pick a global section', 'Backend setting description', 'rey-core'),
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => 'rey-decrease-list-size',
					'id' => '',
				),
				'rey_export' => 'post_id',
				'default_value' => '',
				'gs_type' => 'generic',
				'choices'     => [
					''  => esc_attr__( '- Inherit -', 'rey-core' ),
					'no'  => esc_attr__( '- Disabled -', 'rey-core' ),
				],
			),
			array(
				'key' => 'field_5e24889d03179',
				'label' => esc_html_x('Text', 'Backend setting label', 'rey-core'),
				'name' => 'enable_text_after_add_to_cart',
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => 'rey-decrease-list-size',
					'id' => '',
				),
				'choices' => array(
					'true' => esc_html_x('Show', 'Backend setting label', 'rey-core'),
					'false' => esc_html_x('Hide', 'Backend setting label', 'rey-core'),
					'custom' => esc_html_x('Show Custom Text', 'Backend setting label', 'rey-core'),
				),
				'default_value' => false,
				'allow_null' => 1,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',
			),
			array(
				'key' => 'field_5e2488ad0317a',
				'label' => esc_html_x('Custom text After Add to cart button', 'Backend setting label', 'rey-core'),
				'name' => 'text_after_add_to_cart',
				'type' => 'textarea',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'field_5e24889d03179',
							'operator' => '==',
							'value' => 'custom',
						),
					),
				),
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'default_value' => '',
				'placeholder' => '',
				'maxlength' => '',
				'rows' => '',
				'new_lines' => 'wpautop',
			),
		);

		return [
			'key' => 'group_5d4ff536a2684',
			'title' => 'Product settings',
			'fields' => $product_settings,
			'location' => array(
				array(
					array(
						'param' => 'post_type',
						'operator' => '==',
						'value' => 'product',
					),
				),
			),
			'menu_order' => 0,
			'position' => 'normal',
			'style' => 'default',
			'label_placement' => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen' => '',
			'active' => true,
			'description' => '',

		];

	}

	public function f__page_settings(){
		return [
			'key' => 'group_5c4ad0bd35b33',
			'title' => 'Page Settings',
			'fields' => array(
				array(
					'key' => 'field_5c4ad13830432',
					'label' => esc_html_x('General', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c4ad1920206f',
					'label' => esc_html_x('Title Display', 'Backend setting label', 'rey-core'),
					'name' => 'title_display',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1',
						'id' => '',
					),
					'choices' => array(
						'show' => esc_html_x('Show', 'Backend setting label', 'rey-core'),
						'hide' => esc_html_x('Hide', 'Backend setting label', 'rey-core'),
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ca5a644e6fc2',
					'label' => esc_html_x('Page Cover', 'Backend setting label', 'rey-core'),
					'name' => 'page_cover',
					'type' => 'select',
					'instructions' => esc_html_x('Enable/disable or change the pre-assigned page cover in Customizer > Page Cover.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
					'rey_export' => 'post_id',
				),
				array(
					'key' => 'field_5db54e5bb0124',
					'label' => esc_html_x('Body CSS Classes', 'Backend setting label', 'rey-core'),
					'name' => 'rey_body_class',
					'type' => 'text',
					'instructions' => esc_html_x('Add CSS classes that will be appended to the body tag\'s class attribute.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => '',
					'prepend' => '',
					'append' => '',
					'maxlength' => '',
				),
				array(
					'key' => 'field_5c4ad17030433',
					'label' => esc_html_x('Header', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c4bfd132e0db',
					'label' => esc_html_x('Header Layout', 'Backend setting label', 'rey-core'),
					'name' => 'header_layout_type',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
						'none' => esc_html_x('None', 'Backend setting label', 'rey-core'),
						'default' => esc_html_x('Default Header (Non-Elementor)', 'Backend setting label', 'rey-core'),
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
					'rey_export' => 'post_id',
				),
				array(
					'key' => 'field_5c77bde02960c',
					'label' => esc_html_x('Header Text', 'Backend setting label', 'rey-core'),
					'name' => 'header_text_color',
					'type' => 'color_picker',
					'instructions' => esc_html_x('This will override the Header\'s text color. Please know that some elements may not be colored properly, so in such cases it\'s probably recommended to customize the colors right into the Header Global Section you\'re currently using.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4bfd132e0db',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'enable_opacity' => 1,
					'default_value' => '',
				),
				array(
					'key' => 'field_5c4d67245ab37',
					'label' => esc_html_x('Header Position', 'Backend setting label', 'rey-core'),
					'name' => 'header_position',
					'type' => 'select',
					'instructions' => sprintf(esc_html_x('Select how the header will position itself on top. This option will override the one selected in Customizer > Header > General. Read more about <a href="%s" target="_blank">Header Positions and Overlapping Content</a>', 'Backend setting description', 'rey-core'), reycore__support_url('kb/header-positions-and-overlapping-content/')),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4bfd132e0db',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
						'rel' => esc_html_x('Default (Relative)', 'Backend setting label', 'rey-core'),
						'absolute' => esc_html_x('Absolute (over content)', 'Backend setting label', 'rey-core'),
						'fixed' => esc_html_x('Fixed (Sticky and over content)', 'Backend setting label', 'rey-core'),
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5df93bde34670',
					'label' => esc_html_x('Overlap Content', 'Backend setting label', 'rey-core'),
					'name' => 'header_fixed_overlap',
					'type' => 'select',
					'instructions' => esc_html_x('If enabled, the header will behave like "relative" header. This option will override the global option set in Customizer > Header > General > Header Position -	Overlap Content.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'fixed',
							),
						),
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'absolute',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1 rey-acf-responsive --desktop --dependent',
						'id' => '',
					),
					'choices' => array(
						'true' => esc_html_x('Yes', 'Backend setting label', 'rey-core'),
						'false' => esc_html_x('No', 'Backend setting label', 'rey-core'),
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5e00c7041717e',
					'label' => esc_html_x('Overlap Content (tablet)', 'Backend setting label', 'rey-core'),
					'name' => 'header_fixed_overlap_tablet',
					'type' => 'select',
					'instructions' => esc_html_x('If enabled, the header will behave like "relative" header. This option will override the global option set in Customizer > Header > General > Header Position -	Overlap Content.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'fixed',
							),
						),
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'absolute',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1 rey-acf-responsive --tablet	--dependent',
						'id' => '',
					),
					'choices' => array(
						'true' => 'Yes',
						'false' => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5e00c7361717f',
					'label' => esc_html_x('Overlap Content (mobile)', 'Backend setting label', 'rey-core'),
					'name' => 'header_fixed_overlap_mobile',
					'type' => 'select',
					'instructions' => esc_html_x('If enabled, the header will behave like "relative" header. This option will override the global option set in Customizer > Header > General > Header Position -	Overlap Content.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'fixed',
							),
						),
						array(
							array(
								'field' => 'field_5c4d67245ab37',
								'operator' => '==',
								'value' => 'absolute',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1 rey-acf-responsive --mobile	--dependent',
						'id' => '',
					),
					'choices' => array(
						'true' => 'Yes',
						'false' => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5da5d16549bc4',
					'label' => esc_html_x('Custom Logo', 'Backend setting label', 'rey-core'),
					'name' => 'custom_logo',
					'type' => 'image',
					'instructions' => esc_html_x('Add an image here if you want this page to display a different logo image. If using a Header - Global Section, make sure the Header - Logo element doesn\'t have "Override settings" option enabled.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'return_format' => 'id',
					'preview_size' => 'thumbnail',
					'library' => 'all',
					'min_width' => '',
					'min_height' => '',
					'min_size' => '',
					'max_width' => '',
					'max_height' => '',
					'max_size' => '',
					'mime_types' => '',
				),
				array(
					'key' => 'field_5da5d1d449bc5',
					'label' => esc_html_x('Custom Mobile Logo', 'Backend setting label', 'rey-core'),
					'name' => 'logo_mobile',
					'type' => 'image',
					'instructions' => esc_html_x('Add an image here if you want this page to display a different *mobile* logo image. If using a Header - Global Section, make sure the Header - Logo element doesn\'t have "Override settings" option enabled.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'return_format' => 'id',
					'preview_size' => 'thumbnail',
					'library' => 'all',
					'min_width' => '',
					'min_height' => '',
					'min_size' => '',
					'max_width' => '',
					'max_height' => '',
					'max_size' => '',
					'mime_types' => '',
				),
				array(
					'key' => 'field_5c4ae4aa46da7',
					'label' => esc_html_x('Footer', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c4ae4ba46da8',
					'label' => esc_html_x('Footer Layout', 'Backend setting label', 'rey-core'),
					'name' => 'footer_layout_type',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
					'rey_export' => 'post_id',
				),
				array(
					'key' => 'field_5ecd3bb346234',
					'label' => esc_html_x('Page layout', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5ecd3bc846235',
					'label' => esc_html_x('Container width', 'Backend setting label', 'rey-core'),
					'name' => 'custom_container_width',
					'type' => 'select',
					'instructions' => esc_html_x('Select the page container width.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
						'px' => esc_html_x('Pixels (px)', 'Backend setting label', 'rey-core'),
						'vw' => esc_html_x('Viewport Width (vw)', 'Backend setting label', 'rey-core'),
						'full' => esc_html_x('Full Width', 'Backend setting label', 'rey-core'),
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ecd3ca546236',
					'label' => esc_html_x('Container Width (px)', 'Backend setting label', 'rey-core'),
					'name' => 'container_width_px',
					'type' => 'range',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ecd3bc846235',
								'operator' => '==',
								'value' => 'px',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'default_value' => 1440,
					'min' => 480,
					'max' => 2560,
					'step' => '',
					'prepend' => '',
					'append' => 'px',
				),
				array(
					'key' => 'field_5ecd3d1946237',
					'label' => esc_html_x('Container Width (vw)', 'Backend setting label', 'rey-core'),
					'name' => 'container_width_vw',
					'type' => 'range',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ecd3bc846235',
								'operator' => '==',
								'value' => 'vw',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'default_value' => 90,
					'min' => 1,
					'max' => 100,
					'step' => '',
					'prepend' => '',
					'append' => 'vw',
				),
				array(
					'key' => 'field_5ecd435aadec0',
					'label' => esc_html_x('Apply only to main content', 'Backend setting label', 'rey-core'),
					'name' => 'apply_only_to_main_content',
					'type' => 'true_false',
					'instructions' => esc_html_x('If enabled, this container width will only be applied to the main content (excluding header, footer etc.)', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ecd3bc846235',
								'operator' => '!=empty',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 0,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
				array(
					'key' => 'field_5dbd36274ec3a',
					'label' => esc_html_x('Container Horizontal Margins', 'Backend setting label', 'rey-core'),
					'name' => 'container_spacing',
					'type' => 'number',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => '',
					'prepend' => '',
					'append' => 'px',
					'min' => '',
					'max' => '',
					'step' => '',
				),
				array(
					'key' => 'field_5ecd3daa46239',
					'label' => esc_html_x('Container Vertical Padding', 'Backend setting label', 'rey-core'),
					'name' => 'content_padding',
					'type' => 'group',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-3 rey-acf-dimensions',
						'id' => '',
					),
					'layout' => 'table',
					'sub_fields' => array(
						array(
							'key' => 'field_5ecd3db74623a',
							'label' => esc_html_x('Top', 'Backend setting label', 'rey-core'),
							'name' => 'top',
							'type' => 'number',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => 'px',
							'min' => '',
							'max' => '',
							'step' => '',
						),
						array(
							'key' => 'field_5ecd3ddd4623c',
							'label' => esc_html_x('Bottom', 'Backend setting label', 'rey-core'),
							'name' => 'bottom',
							'type' => 'number',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => 'px',
							'min' => '',
							'max' => '',
							'step' => '',
						),
					),
				),
				array(
					'key' => 'field_5c4c848308731',
					'label' => esc_html_x('Advanced', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c4cba8c273ad',
					'label' => esc_html_x('Global Sections', 'Backend setting label', 'rey-core'),
					'name' => 'global_sections',
					'type' => 'repeater',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-acf-repeater-list',
						'id' => '',
					),
					'collapsed' => '',
					'min' => 0,
					'max' => 0,
					'layout' => 'table',
					'button_label' => 'Add Global Section',
					'sub_fields' => array(
						array(
							'key' => 'field_5c4cbbb2273af',
							'label' => esc_html_x('Generic Sections List', 'Backend setting label', 'rey-core'),
							'name' => 'global_sections_id',
							'type' => 'select',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => 'rey-decrease-list-size',
								'id' => '',
							),
							'choices' => array(
							),
							'default_value' => false,
							'allow_null' => 1,
							'multiple' => 0,
							'ui' => 1,
							'ajax' => 0,
							'return_format' => 'value',
							'placeholder' => '',
							'rey_export' => 'post_id',
						),
						array(
							'key' => 'field_5c4cbb49273ae',
							'label' => esc_html_x('Position', 'Backend setting label', 'rey-core'),
							'name' => 'global_sections_hook',
							'type' => 'select',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => 'rey-decrease-list-size',
								'id' => '',
							),
							'choices' => array(
								'before_site_wrapper' => esc_html_x('Before Site Wrapper', 'Backend setting label', 'rey-core'),
								'before_header' => esc_html_x('Before Header', 'Backend setting label', 'rey-core'),
								'after_header' => esc_html_x('After Header', 'Backend setting label', 'rey-core'),
								'before_site_container' => esc_html_x('Before Site Container', 'Backend setting label', 'rey-core'),
								'after_site_container' => esc_html_x('After Site Container', 'Backend setting label', 'rey-core'),
								'before_footer' => esc_html_x('Before Footer', 'Backend setting label', 'rey-core'),
								'after_footer' => esc_html_x('After Footer', 'Backend setting label', 'rey-core'),
								'after_site_wrapper' => esc_html_x('After Site Wrapper', 'Backend setting label', 'rey-core'),
								'wp_body_open' => esc_html_x('Body Start (wp_body_open)', 'Backend setting label', 'rey-core'),
								'wp_footer' => esc_html_x('Body End (wp_footer)', 'Backend setting label', 'rey-core'),
							),
							'default_value' => false,
							'allow_null' => 1,
							'multiple' => 0,
							'ui' => 1,
							'ajax' => 0,
							'return_format' => 'value',
							'placeholder' => '',
						),
					),
				),
				array(
					'key' => 'field_5c4c84ef08732',
					'label' => esc_html_x('Remove Pre-assigned Global Sections', 'Backend setting label', 'rey-core'),
					'name' => 'remove_global_sections',
					'type' => 'select',
					'instructions' => esc_html_x('Select if you want to remove the global settings already assigned site-wide in Customizer\'s options > General settings > Global Sections. Hold CTRL key to select multiple.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => array(
					),
					'allow_null' => 1,
					'multiple' => 1,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5c4ad2d9a592c',
					'label' => esc_html_x('Redirect', 'Backend setting label', 'rey-core'),
					'name' => 'general_redirect',
					'type' => 'true_false',
					'instructions' => esc_html_x('Make this post/page a redirect.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 0,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
				array(
					'key' => 'field_5c4ad2fea592d',
					'label' => esc_html_x('Redirect URL', 'Backend setting label', 'rey-core'),
					'name' => 'redirect_url',
					'type' => 'link',
					'instructions' => esc_html_x('Enter the URL.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5c4ad2d9a592c',
								'operator' => '==',
								'value' => '1',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'return_format' => 'url',
				),
				array(
					'key' => 'field_5ed770b88a773',
					'label' => esc_html_x('Sticky Global Section - Top', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs',
					'type' => 'select',
					'instructions' => esc_html_x('Select to add or override the sticky top global section option pre-defined in Customizer > General > Sticky Global Sections', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-3',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
					'rey_export' => 'post_id',
				),
				array(
					'key' => 'field_5ed77352a54b3',
					'label' => esc_html_x('Activation Offset', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs_offset',
					'type' => 'text',
					'instructions' => esc_html_x('At what distance from the top edge should be displayed. Add value in pixels eg: 100 or 200px etc. or a unique id selector, eg: #my_unique_element . If empty it\'ll be triggered after the site-header exists viewport.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-3 --dependent',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => '',
					'prepend' => '',
					'append' => '',
					'maxlength' => '',
				),
				array(
					'key' => 'field_5ed773c0a54b4',
					'label' => esc_html_x('Text Color', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs_color',
					'type' => 'color_picker',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => ' --dependent',
						'id' => '',
					),
					'enable_opacity' => 1,
					'default_value' => '',
				),
				array(
					'key' => 'field_5ed773eca54b5',
					'label' => esc_html_x('Background Color', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs_bg_color',
					'type' => 'color_picker',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => ' --dependent',
						'id' => '',
					),
					'enable_opacity' => 1,
					'default_value' => '',
				),
				array(
					'key' => 'field_60e5e45cc8f30',
					'label' => esc_html_x('Select visibility', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs_hide_devices',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => ' --dependent',
						'id' => '',
					),
					'choices' => array(
						'desktop' => esc_html_x('Hide on desktop', 'Backend setting label', 'rey-core'),
						'tablet' => esc_html_x('Hide on tablet', 'Backend setting label', 'rey-core'),
						'mobile' => esc_html_x('Hide on mobile', 'Backend setting label', 'rey-core'),
					),
					'default_value' => array(
					),
					'allow_null' => 1,
					'multiple' => 1,
					'ui' => 1,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ed7747da54b7',
					'label' => esc_html_x('Add close button?', 'Backend setting label', 'rey-core'),
					'name' => 'top_sticky_gs_close',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed770b88a773',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1 --dependent',
						'id' => '',
					),
					'choices' => array(
						'yes' => 'Yes',
						'no' => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ed771438a774',
					'label' => esc_html_x('Sticky Global Section - Bottom', 'Backend setting label', 'rey-core'),
					'name' => 'bottom_sticky_gs',
					'type' => 'select',
					'instructions' => esc_html_x('Select to add or override the sticky top global section option pre-defined in Customizer > General > Sticky Global Sections', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-3',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
					'rey_export' => 'post_id',
				),
				array(
					'key' => 'field_5ed77694922fa',
					'label' => esc_html_x('Activation Offset', 'Backend setting label', 'rey-core'),
					'name' => 'bottom_sticky_gs_offset',
					'type' => 'text',
					'instructions' => esc_html_x('At what distance from the top edge should be displayed. Add value in pixels eg: 100 or 200px etc. or a unique id selector, eg: #my_unique_element . If empty it\'ll be triggered after the site-header exists viewport.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-3 --dependent',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => '',
					'prepend' => '',
					'append' => '',
					'maxlength' => '',
				),
				array(
					'key' => 'field_6e45f0cc8e530',
					'label' => esc_html_x('Select visibility', 'Backend setting label', 'rey-core'),
					'name' => 'bottom_sticky_gs_hide_devices',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => ' --dependent',
						'id' => '',
					),
					'choices' => array(
						'desktop' => esc_html_x('Hide on desktop', 'Backend setting label', 'rey-core'),
						'tablet' => esc_html_x('Hide on tablet', 'Backend setting label', 'rey-core'),
						'mobile' => esc_html_x('Hide on mobile', 'Backend setting label', 'rey-core'),
					),
					'default_value' => array(
					),
					'allow_null' => 1,
					'multiple' => 1,
					'ui' => 1,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ed776c2922fc',
					'label' => esc_html_x('Add close button?', 'Backend setting label', 'rey-core'),
					'name' => 'bottom_sticky_gs_close',
					'type' => 'select',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => '',
							),
							array(
								'field' => 'field_5ed771438a774',
								'operator' => '!=',
								'value' => 'none',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1 --dependent',
						'id' => '',
					),
					'choices' => array(
						'yes' => 'Yes',
						'no' => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_6031467caec55',
					'label' => '',
					'name' => '',
					'type' => 'message',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '<h3>Assets</h3>',
					'new_lines' => 'wpautop',
					'esc_html' => 0,
				),
				array(
					'key' => 'field_603146f7aec56',
					'label' => esc_html_x('Disable WordPress Blocks Styles', 'Backend setting label', 'rey-core'),
					'name' => 'perf__disable_wpblock',
					'type' => 'select',
					'instructions' => esc_html_x('Will disable WordPress\'s built-in Gutenberg editor styles. Enable this option if you don\'t use blocks throughout this page.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1',
						'id' => '',
					),
					'choices' => array(
						1 => 'Yes',
						0 => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
				array(
					'key' => 'field_60314765aec57',
					'label' => esc_html_x('Disable WooCommerce Blocks Styles', 'Backend setting label', 'rey-core'),
					'name' => 'perf__disable_wcblock',
					'type' => 'select',
					'instructions' => esc_html_x('Will disable WordPress\'s built-in Gutenberg editor styles. Enable this option if you don\'t use blocks throughout this page.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--size-1',
						'id' => '',
					),
					'choices' => array(
						1 => 'Yes',
						0 => 'No',
					),
					'default_value' => false,
					'allow_null' => 1,
					'multiple' => 0,
					'ui' => 0,
					'return_format' => 'value',
					'ajax' => 0,
					'placeholder' => '',
				),
			),
			'location' => array(
			),
			'menu_order' => 0,
			'position' => 'normal',
			'style' => 'default',
			'label_placement' => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen' => '',
			'active' => true,
			'description' => 'These settings apply for this post/page/taxonomies.',

		];
	}

	public function f__theme_settings(){
		return [
			'key' => 'group_5c990a758cfda',
			'title' => 'Theme Settings',
			'fields' => array(
				array(
					'key' => 'field_5d3afdfd482f1',
					'label' => esc_html_x('Font Settings', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c9e384a43424',
					'label' => '',
					'name' => '',
					'type' => 'message',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '<h1>Font Settings</h1><p>Adjust your website\'s typography settings. To customize frontend\'s typography, head over to Customizer > General > Typography.</p>',
					'new_lines' => 'wpautop',
					'esc_html' => 0,
				),
				array(
					'key' => 'field_642d2804385d3',
					'label' => '',
					'name' => '',
					'type' => 'message',
					'wrapper' => array(
						'class' => '--title-h3',
					),
					'message' => '<h3>Google Fonts</h3>',
					'esc_html' => 0,
				),

				array(
					'key' => 'field_642d2695385d0',
					'label' => esc_html_x('Optimisations', 'Backend setting label', 'rey-core'),
					'name' => 'font_optimisations',
					'aria-label' => '',
					'type' => 'group',
					'instructions' => esc_html_x('Improve the performance of your website by optimizing what and how Google Fonts load in your website.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'layout' => 'table',
					'sub_fields' => array(
						array(
							'key' => 'field_642d2752385d1',
							'label' => esc_html_x('Font Weights', 'Backend setting label', 'rey-core'),
							'name' => 'weights',
							'aria-label' => '',
							'type' => 'select',
							'instructions' => esc_html_x('Specify which font weights to load from Google Fonts. The less weights are loaded the faster the website.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'choices' => \ReyCore\Webfonts::get_default_weights(true),
							'default_value' => [],
							'return_format' => 'value',
							'multiple' => 1,
							'allow_null' => 0,
							'ui' => 1,
							'ajax' => 0,
							'placeholder' => '',
						),
						array(
							'key' => 'field_642d27a3385d2',
							'label' => esc_html_x('Font Subsets', 'Backend setting label', 'rey-core'),
							'name' => 'subsets',
							'aria-label' => '',
							'type' => 'select',
							'instructions' => esc_html_x('Specify which font subsets to load from Google Fonts. Subsets are usually needed for non-latin characters (like Cyrillic, Greek, Hebrew etc. ). The less subsets are loaded the faster the website.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'choices' => \ReyCore\Webfonts::get_default_subsets(true),
							'default_value' => [],
							'return_format' => 'value',
							'multiple' => 1,
							'allow_null' => 0,
							'ui' => 1,
							'ajax' => 0,
							'placeholder' => '',
						),
					),
				),

				array(
					'key' => 'field_5c997501aef80',
					'label' => esc_html_x('Self-host Google Fonts', 'Backend setting label', 'rey-core'),
					'name' => 'self_host_fonts',
					'type' => 'true_false',
					'instructions' => esc_html_x('This option will store the Google Fonts on your server, resulting in potentially faster rendering.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 1,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
			array(
				'key' => 'field_6428435dd2803',
				'label' => '',
				'name' => '',
				'type' => 'message',
				'wrapper' => array(
					'class' => '--title-h3',
				),
				'message' => '<h3>Adobe Fonts</h3>',
				'esc_html' => 0,
			),
				array(
					'key' => 'field_5caf1fa73b466',
					'label' => esc_html_x('Adobe Fonts Project ID (Typekit)', 'Backend setting label', 'rey-core'),
					'name' => 'adobe_fonts_project_id',
					'type' => 'text',
					'instructions' => esc_html_x('Create a new <a href="https://fonts.adobe.com/my_fonts#web_projects-section" target="_blank">Adobe Font Web Project</a> and paste here its project ID. After saving this page, you\'ll be able to notice the new fonts in the fonts lists of Customizer and Elementor.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => 'eg: pa2xijk',
					'prepend' => '',
					'append' => '',
					'maxlength' => 10,
				),
			array(
				'key' => 'field_642d4385280d3',
				'label' => '',
				'name' => '',
				'type' => 'message',
				'wrapper' => array(
					'class' => '--title-h3',
				),
				'message' => '<h3>Custom Fonts</h3>',
				'esc_html' => 0,
			),
				array(
					'key' => 'field_5ef73f139c0a1',
					'label' => esc_html_x('Custom Fonts', 'Backend setting label', 'rey-core'),
					'name' => 'custom_fonts',
					'type' => 'repeater',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '--acf-repeater-row',
						'id' => '',
					),
					'collapsed' => '',
					'min' => 0,
					'max' => 0,
					'layout' => 'table',
					'button_label' => 'Add Custom Font',
					'sub_fields' => array(
						array(
							'key' => 'field_5ef73f449c0a2',
							'label' => esc_html_x('Name', 'Backend setting label', 'rey-core'),
							'name' => 'name',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_5ef0a2449f73c',
							'label' => esc_html_x('Weight', 'Backend setting label', 'rey-core'),
							'name' => 'weight',
							'type' => 'text',
							'instructions' => esc_html_x('Variable fonts support multiple weights', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_5ef73f579c0a3',
							'label' => 'Font .woff2',
							'name' => 'font_woff2',
							'type' => 'file',
							'instructions' => esc_html_x('Upload the font\'s woff2 file.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'return_format' => 'id',
							'library' => 'all',
							'min_size' => '',
							'max_size' => '',
							'mime_types' => '.woff2',
						),
						array(
							'key' => 'field_5ef73fab9c0a4',
							'label' => 'Font .woff',
							'name' => 'font_woff',
							'type' => 'file',
							'instructions' => esc_html_x('Upload the font\'s woff file.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'return_format' => 'url',
							'library' => 'all',
							'min_size' => '',
							'max_size' => '',
							'mime_types' => '.woff',
						),
						array(
							'key' => 'field_5ef73fae9c0a5',
							'label' => 'Font .ttf',
							'name' => 'font_ttf',
							'type' => 'file',
							'instructions' => esc_html_x('Upload the font\'s ttf file.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'return_format' => 'url',
							'library' => 'all',
							'min_size' => '',
							'max_size' => '',
							'mime_types' => '.ttf',
						),
						array(
							'key' => 'field_5ef73fbf9c0a8',
							'label' => 'Font .otf',
							'name' => 'font_otf',
							'type' => 'file',
							'instructions' => esc_html_x('Upload the font\'s otf file.', 'Backend setting description', 'rey-core'),
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'return_format' => 'url',
							'library' => 'all',
							'min_size' => '',
							'max_size' => '',
							'mime_types' => '.otf',
						),
						array(
							'key' => 'field_5ef740139c0a9',
							'label' => esc_html_x('Font Display', 'Backend setting label', 'rey-core'),
							'name' => 'font_display',
							'type' => 'select',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'choices' => array(
								'auto' => esc_html_x('Auto', 'Backend setting label', 'rey-core'),
								'block' => esc_html_x('Block', 'Backend setting label', 'rey-core'),
								'swap' => esc_html_x('Swap', 'Backend setting label', 'rey-core'),
								'fallback' => esc_html_x('Fallback', 'Backend setting label', 'rey-core'),
								'optional' => esc_html_x('Optional', 'Backend setting label', 'rey-core'),
							),
							'default_value' => 'auto',
							'allow_null' => 0,
							'multiple' => 0,
							'ui' => 0,
							'return_format' => 'value',
							'ajax' => 0,
							'placeholder' => '',
						),
					),
				),

				array(
					'key' => 'field_5d3afe12f2648',
					'label' => esc_html_x('Integrations', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),

				array(
					'key' => 'field_5c9e3a1250024',
					'label' => '',
					'name' => '',
					'type' => 'message',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '<h1>Integration Settings</h1>',
					'new_lines' => 'wpautop',
					'esc_html' => 0,
				),

				array(
					'key' => 'field_5db54e01245bb',
					'label' => esc_html_x('Instagram Access Token', 'Backend setting label', 'rey-core'),
					'name' => 'ig_access_token',
					'type' => 'text',
					'instructions' => sprintf(_x('Access Token used for Instagram Image Feeds. Please read <strong><a href="%s" target="_blank">this article</a></strong> to find out how to generate an Instagram Access Token.', 'Backend setting description', 'rey-core'), reycore__support_url('kb/add-instagram-feeds/#generate-access-tokens') ),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'default_value' => '',
					'placeholder' => 'example: IGQVJYTGpIVHJyNAsdEhDLUtVZA0wxWG53cHp2UzMUJib3hjcU1YQkE0zTVlkV9uRS1sSzFfX05zOUNES1ArNTRoeXBDOEdScnFJWWZABSE5STE9fS1N0ZAiNHNCM3VXZAkZYM1NmQyYkJB',
					'prepend' => '',
					'append' => '',
					'maxlength' => '',
				),

				array(
					'key' => 'field_5d3afe16482f2',
					'label' => esc_html_x('Misc. Settings', 'Backend setting label', 'rey-core'),
					'name' => '',
					'type' => 'tab',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'placement' => 'top',
					'endpoint' => 0,
				),
				array(
					'key' => 'field_5c9e3a0241250',
					'label' => '',
					'name' => '',
					'type' => 'message',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '<h1>Miscellaneous Settings</h1>',
					'new_lines' => 'wpautop',
					'esc_html' => 0,
				),
				array(
					'key' => 'field_5c991011d16cd',
					'label' => esc_html_x('ACF fields panel', 'Backend setting label', 'rey-core'),
					'name' => 'acf_fields_panel',
					'type' => 'true_false',
					'instructions' => esc_html_x('Enable or disable the custom fields panel (ACF Panel) in the WordPress backend menu.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 1,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
				array(
					'key' => 'field_5c01d16c1991d',
					'label' => esc_html_x('Widgets Page - Blocks Layout', 'Backend setting label', 'rey-core'),
					'name' => 'rey_widgets_blocks_layout',
					'type' => 'true_false',
					'instructions' => esc_html_x('Enable Widgets page modern WP 5.8+ blocks layout.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 0,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
				array(
					'key' => 'field_5ebea386f5754',
					'label' => esc_html_x('Post types support for "Page Settings"', 'Backend setting label', 'rey-core'),
					'name' => 'singular_settings_support',
					'type' => 'select',
					'instructions' => esc_html_x('Select on which post types to display the "Page Settings" options block.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'choices' => [],
					'allow_custom' => 0,
					'default_value' => [],
					'multiple' => 1,
					'ui' => 1,
					'ajax' => 0,
					'return_format' => 'value',
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ebea386f5755',
					'label' => esc_html_x('Taxonomy support for "Page Settings"', 'Backend setting label', 'rey-core'),
					'name' => 'singular_settings_support_taxonomy',
					'type' => 'select',
					'instructions' => esc_html_x('Select on which taxonomy post types to display the "Page Settings" options block.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'choices' => [],
					'allow_custom' => 0,
					'default_value' => [],
					'multiple' => 1,
					'ui' => 1,
					'ajax' => 0,
					'return_format' => 'value',
					'placeholder' => '',
				),
				array(
					'key' => 'field_5f7c4cfd0fff9',
					'label' => esc_html_x('Page Settings - User Roles', 'Backend setting label', 'rey-core'),
					'name' => 'singular_settings_support_roles',
					'type' => 'select',
					'instructions' => esc_html_x('Select allowed user roles to display the Page settings.', 'Backend setting description', 'rey-core'),
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => 'rey-decrease-list-size',
						'id' => '',
					),
					'choices' => array(
					),
					'default_value' => 'all',
					'allow_null' => 0,
					'multiple' => 0,
					'ui' => 1,
					'ajax' => 0,
					'return_format' => 'value',
					'placeholder' => '',
				),
				array(
					'key' => 'field_5ea3ebf575865',
					'label' => esc_html_x('Hide links in menu', 'Backend setting label', 'rey-core'),
					'name' => 'hide_admin_links',
					'type' => 'select',
					'instructions' => esc_html_x('Select which menu items should be hidden in the side menu, under Management.', 'Backend setting description', 'rey-core'),
					'choices' => [
						'customizer' => esc_html_x('Customizer', 'Backend setting label', 'rey-core'),
						'demo_import' => esc_html_x('Import Demo Sites', 'Backend setting label', 'rey-core'),
						'modules' => esc_html_x('Modules Manager', 'Backend setting label', 'rey-core'),
						'elements' => esc_html_x('Elements Manager', 'Backend setting label', 'rey-core'),
						'plugins' => esc_html_x('Plugins Manager', 'Backend setting label', 'rey-core'),
					],
					'allow_custom' => 0,
					'default_value' => [],
					'multiple' => 1,
					'ui' => 1,
					'ajax' => 0,
					'return_format' => 'value',
					'placeholder' => '',
				),
			),
			'location' => array(
				array(
					array(
						'param' => 'options_page',
						'operator' => '==',
						'value' => 'rey-settings',
					),
				),
			),
			'menu_order' => 0,
			'position' => 'acf_after_title',
			'style' => 'seamless',
			'label_placement' => 'left',
			'instruction_placement' => 'field',
			'hide_on_screen' => array(
				0 => 'permalink',
				1 => 'the_content',
				2 => 'excerpt',
				3 => 'discussion',
				4 => 'comments',
				5 => 'revisions',
				6 => 'slug',
				7 => 'author',
				8 => 'format',
				9 => 'page_attributes',
				10 => 'featured_image',
				11 => 'categories',
				12 => 'tags',
				13 => 'send-trackbacks',
			),
			'active' => true,
			'description' => '',

		];
	}

	public function f__icon_menu_settings(){
		return [
			'key' => 'group_5f0439874af42',
			'title' => 'Icon Menu Settings',
			'fields' => array(
				array(
					'key' => 'field_5f04398752b45',
					'label' => esc_html_x('Menu item Icon', 'Backend setting label', 'rey-core'),
					'name' => 'enable_menu_item_icon',
					'type' => 'true_false',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => 0,
					'wrapper' => array(
						'width' => '',
						'class' => '',
						'id' => '',
					),
					'message' => '',
					'default_value' => 0,
					'ui' => 1,
					'ui_on_text' => '',
					'ui_off_text' => '',
				),
				array(
					'key' => 'field_5f04398752b4c',
					'label' => esc_html_x('Select Icon', 'Backend setting label', 'rey-core'),
					'name' => 'menu_item_icon',
					'type' => 'image',
					'instructions' => '',
					'required' => 0,
					'conditional_logic' => array(
						array(
							array(
								'field' => 'field_5f04398752b45',
								'operator' => '==',
								'value' => '1',
							),
						),
					),
					'wrapper' => array(
						'width' => '',
						'class' => '--dependent',
						'id' => '',
					),
					'return_format' => 'id',
					'preview_size' => 'thumbnail',
					'library' => 'all',
					'min_width' => '',
					'min_height' => '',
					'min_size' => '',
					'max_width' => '',
					'max_height' => '',
					'max_size' => '',
					'mime_types' => '',
				),
			),
			'location' => array(
				array(
					array(
						'param' => 'nav_menu_item',
						'operator' => '==',
						'value' => 'all',
					),
				),
			),
			'menu_order' => 10,
			'position' => 'normal',
			'style' => 'default',
			'label_placement' => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen' => '',
			'active' => true,
			'description' => '',

		];
	}

}

new Fields();
