<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace MediaCloud\Vendor\Symfony\Component\Messenger\Command;
use MediaCloud\Vendor\Psr\Container\ContainerInterface;
use MediaCloud\Vendor\Symfony\Component\Console\Command\Command;
use MediaCloud\Vendor\Symfony\Component\Console\Completion\CompletionInput;
use MediaCloud\Vendor\Symfony\Component\Console\Completion\CompletionSuggestions;
use MediaCloud\Vendor\Symfony\Component\Console\Input\InputArgument;
use MediaCloud\Vendor\Symfony\Component\Console\Input\InputInterface;
use MediaCloud\Vendor\Symfony\Component\Console\Output\OutputInterface;
use MediaCloud\Vendor\Symfony\Component\Console\Style\SymfonyStyle;
use MediaCloud\Vendor\Symfony\Component\Messenger\Transport\SetupableTransportInterface;

/**
 * @author Vincent Touzet <vincent.touzet@gmail.com>
 */
class SetupTransportsCommand extends Command
{
    protected static $defaultName = 'messenger:setup-transports';
    protected static $defaultDescription = 'Prepare the required infrastructure for the transport';

    private $transportLocator;
    private $transportNames;

    public function __construct(ContainerInterface $transportLocator, array $transportNames = [])
    {
        $this->transportLocator = $transportLocator;
        $this->transportNames = $transportNames;

        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->addArgument('transport', InputArgument::OPTIONAL, 'Name of the transport to setup', null)
            ->setDescription(self::$defaultDescription)
            ->setHelp(<<<EOF
The <info>%command.name%</info> command setups the transports:

    <info>php %command.full_name%</info>

Or a specific transport only:

    <info>php %command.full_name% <transport></info>
EOF
            )
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);

        $transportNames = $this->transportNames;
        // do we want to set up only one transport?
        if ($transport = $input->getArgument('transport')) {
            if (!$this->transportLocator->has($transport)) {
                throw new \RuntimeException(sprintf('The "%s" transport does not exist.', $transport));
            }
            $transportNames = [$transport];
        }

        foreach ($transportNames as $id => $transportName) {
            $transport = $this->transportLocator->get($transportName);
            if ($transport instanceof SetupableTransportInterface) {
                $transport->setup();
                $io->success(sprintf('The "%s" transport was set up successfully.', $transportName));
            } else {
                $io->note(sprintf('The "%s" transport does not support setup.', $transportName));
            }
        }

        return 0;
    }

    public function complete(CompletionInput $input, CompletionSuggestions $suggestions): void
    {
        if ($input->mustSuggestArgumentValuesFor('transport')) {
            $suggestions->suggestValues($this->transportNames);

            return;
        }
    }
}
